<?php
session_start();

// Simular una sesión autenticada
$_SESSION['user_id'] = 1;
$_SESSION['user_name'] = 'Test User';
$_SESSION['user_role'] = 'admin';
$_SESSION['authenticated'] = true;

// Incluir configuración de base de datos
require_once 'database/config.php';

// Verificar conexión a la base de datos
try {
    $pdo = new PDO("mysql:host=" . DB_HOST . ";port=" . DB_PORT . ";dbname=" . DB_NAME, DB_USER, DB_PASS);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $db_status = "✅ Conexión exitosa";
} catch (PDOException $e) {
    $db_status = "❌ Error: " . $e->getMessage();
}

// Obtener roles para la prueba
$roles = [];
try {
    $stmt = $pdo->query("SELECT id, display_name, description FROM roles LIMIT 5");
    $roles = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (Exception $e) {
    $roles_error = $e->getMessage();
}
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Test Editar Roles - Sesión Autenticada</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; }
        .status { padding: 10px; margin: 10px 0; border-radius: 5px; }
        .success { background-color: #d4edda; color: #155724; border: 1px solid #c3e6cb; }
        .error { background-color: #f8d7da; color: #721c24; border: 1px solid #f5c6cb; }
        .info { background-color: #d1ecf1; color: #0c5460; border: 1px solid #bee5eb; }
        .test-section { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 5px; }
        button { padding: 8px 15px; margin: 5px; cursor: pointer; }
        .edit-btn { background-color: #007bff; color: white; border: none; border-radius: 3px; }
        .edit-btn:hover { background-color: #0056b3; }
        #results { margin-top: 20px; }
        .result { padding: 10px; margin: 5px 0; border-radius: 3px; }
        .result.success { background-color: #d4edda; }
        .result.error { background-color: #f8d7da; }
    </style>
</head>
<body>
    <h1>🔧 Test Editar Roles - Sesión Autenticada</h1>
    
    <div class="test-section">
        <h2>Estado del Sistema</h2>
        <div class="status info">
            <strong>Sesión:</strong> ✅ Usuario autenticado (ID: <?php echo $_SESSION['user_id']; ?>, Rol: <?php echo $_SESSION['user_role']; ?>)
        </div>
        <div class="status <?php echo strpos($db_status, '✅') !== false ? 'success' : 'error'; ?>">
            <strong>Base de Datos:</strong> <?php echo $db_status; ?>
        </div>
    </div>

    <div class="test-section">
        <h2>Roles Disponibles</h2>
        <?php if (!empty($roles)): ?>
            <table border="1" cellpadding="10" cellspacing="0" style="width: 100%; border-collapse: collapse;">
                <thead>
                    <tr>
                        <th>ID</th>
                        <th>Nombre</th>
                        <th>Descripción</th>
                        <th>Acciones</th>
                    </tr>
                </thead>
                <tbody>
                    <?php foreach ($roles as $role): ?>
                    <tr>
                        <td><?php echo htmlspecialchars($role['id']); ?></td>
                        <td><?php echo htmlspecialchars($role['display_name']); ?></td>
                        <td><?php echo htmlspecialchars($role['description']); ?></td>
                        <td>
                            <button class="edit-btn" onclick="testEditRole(<?php echo $role['id']; ?>)">
                                Editar
                            </button>
                        </td>
                    </tr>
                    <?php endforeach; ?>
                </tbody>
            </table>
        <?php else: ?>
            <div class="status error">
                <strong>Error:</strong> No se pudieron cargar los roles.
                <?php if (isset($roles_error)): ?>
                    <br>Detalle: <?php echo htmlspecialchars($roles_error); ?>
                <?php endif; ?>
            </div>
        <?php endif; ?>
    </div>

    <div class="test-section">
        <h2>Pruebas Directas de Endpoints</h2>
        <button onclick="testGetRoleDetails(1)">Test get_role_details (ID: 1)</button>
        <button onclick="testGetPermissions()">Test get_permissions</button>
        <button onclick="testGetUsers()">Test get_users</button>
    </div>

    <div id="results">
        <h2>Resultados de Pruebas</h2>
    </div>

    <script>
        function addResult(message, isSuccess = true) {
            const results = document.getElementById('results');
            const div = document.createElement('div');
            div.className = `result ${isSuccess ? 'success' : 'error'}`;
            div.innerHTML = `<strong>${new Date().toLocaleTimeString()}:</strong> ${message}`;
            results.appendChild(div);
        }

        async function testEditRole(roleId) {
            addResult(`🔄 Iniciando prueba de editar rol ID: ${roleId}`, true);
            
            try {
                // Simular exactamente lo que hace la función editRole original
                const response = await fetch(`modules/roles/actions.php?action=get_role_details&role_id=${encodeURIComponent(roleId)}`);

                addResult(`📡 Respuesta recibida - Status: ${response.status} (${response.statusText})`, response.ok);
                
                const contentType = response.headers.get('content-type');
                addResult(`📋 Content-Type: ${contentType}`, true);

                const responseText = await response.text();
                addResult(`📄 Contenido de respuesta (primeros 200 chars): ${responseText.substring(0, 200)}...`, true);

                if (contentType && contentType.includes('application/json')) {
                    try {
                        const data = JSON.parse(responseText);
                        addResult(`✅ JSON válido recibido: ${JSON.stringify(data, null, 2)}`, true);
                        
                        // Simular exactamente el manejo de errores de la función original
                        if (data.error || data.success === false) {
                            addResult(`⚠️ Error del servidor: ${data.error || data.message || 'No autorizado'}`, false);
                            addResult(`🔍 DIAGNÓSTICO: Este es el error que causa el problema en editRole()`, false);
                        } else if (data.role) {
                            addResult(`🎉 Éxito: Datos del rol obtenidos correctamente`, true);
                        } else {
                            addResult(`⚠️ Respuesta sin datos de rol`, false);
                        }
                    } catch (jsonError) {
                        addResult(`❌ Error al parsear JSON: ${jsonError.message}`, false);
                        addResult(`🔍 DIAGNÓSTICO: Este es el error "Unexpected token" que ves`, false);
                    }
                } else {
                    addResult(`❌ Respuesta no es JSON. Posible HTML o error del servidor`, false);
                }

            } catch (error) {
                addResult(`❌ Error de red: ${error.message}`, false);
            }
        }

        async function testGetRoleDetails(roleId) {
            try {
                // Cambiar a GET para coincidir con lo que espera el endpoint
                const response = await fetch(`modules/roles/actions.php?action=get_role_details&role_id=${roleId}`, {
                    method: 'GET'
                });

                const text = await response.text();
                addResult(`get_role_details: ${response.status} - ${text.substring(0, 100)}...`, response.ok);
            } catch (error) {
                addResult(`get_role_details error: ${error.message}`, false);
            }
        }

        async function testGetPermissions() {
            try {
                const response = await fetch('modules/roles/actions.php?action=get_permissions&role_id=1');
                const text = await response.text();
                addResult(`get_permissions (role_id=1): ${response.status} - ${text.substring(0, 100)}...`, response.ok);
            } catch (error) {
                addResult(`get_permissions error: ${error.message}`, false);
            }
        }

        async function testGetUsers() {
            try {
                const response = await fetch('modules/roles/actions.php?action=get_users&role_id=1');
                const text = await response.text();
                addResult(`get_users (role_id=1): ${response.status} - ${text.substring(0, 100)}...`, response.ok);
            } catch (error) {
                addResult(`get_users error: ${error.message}`, false);
            }
        }

        // Agregar mensaje inicial
        addResult('🚀 Sistema de pruebas iniciado. Haz clic en "Editar" para probar la funcionalidad.', true);
    </script>
</body>
</html>