<?php
/**
 * API Endpoint: Candles (OHLC)
 * Devuelve velas históricas para un símbolo (por defecto EURUSD) en 1 minuto.
 * El frontend agrega a otros timeframes desde el buffer de 1M.
 */

session_start();
header('Content-Type: application/json');

require_once __DIR__ . '/../../../database/connection.php';
require_once __DIR__ . '/../finage_api.php';

try {
    $symbol = isset($_GET['symbol']) ? strtoupper(trim($_GET['symbol'])) : 'EURUSD';
    $timeframe = isset($_GET['timeframe']) ? intval($_GET['timeframe']) : 1; // minutos
    $limit = isset($_GET['limit']) ? intval($_GET['limit']) : 200;
    if ($limit <= 0 || $limit > 2000) $limit = 200;
    $months = isset($_GET['months']) ? intval($_GET['months']) : 0;
    $multiply = isset($_GET['multiply']) ? intval($_GET['multiply']) : 1; // tamaño del bloque
    $unit = isset($_GET['unit']) ? strtolower(trim($_GET['unit'])) : 'minute';
    if (!in_array($unit, ['minute','hour','day'])) $unit = 'minute';

    $finage = getFinageAPI();
    $candles = [];
    $logTag = '[candles.php]';
    // 1) Intentar agregados por rango: si months>0 usar desde fecha-"months" hasta hoy, con multiply/unit
    try {
        $nowUtc = new DateTime('now', new DateTimeZone('UTC'));
        $fromUtc = clone $nowUtc;
        if ($months > 0) { $fromUtc->modify("-{$months} months"); } else { $fromUtc->setTime(0,0,0); }
        $m = max(1, $multiply);
        $u = $unit;
        // Si el usuario solicita 1M en 3 meses, fragmentar por semanas para evitar límites
        $cursor = clone $fromUtc;
        while ($cursor <= $nowUtc) {
            $chunkFrom = $cursor->format('Y-m-d');
            $chunkToDate = clone $cursor; $chunkToDate->modify('+7 days');
            if ($chunkToDate > $nowUtc) $chunkToDate = clone $nowUtc;
            $chunkTo = $chunkToDate->format('Y-m-d');
            $rangeData = $finage->getAggregatesRange($symbol, $m, $u, $chunkFrom, $chunkTo, 'forex');
            error_log("$logTag range {$u} m={$m} from=$chunkFrom to=$chunkTo count=" . (is_array($rangeData) ? count($rangeData) : 0));
            if (is_array($rangeData) && count($rangeData) > 0) {
                foreach ($rangeData as $row) {
                    $x = isset($row['time']) ? intval($row['time']) * 1000 : (isset($row['t']) ? intval($row['t']) : null);
                    $o = isset($row['open']) ? floatval($row['open']) : (isset($row['o']) ? floatval($row['o']) : null);
                    $h = isset($row['high']) ? floatval($row['high']) : (isset($row['h']) ? floatval($row['h']) : null);
                    $l = isset($row['low']) ? floatval($row['low']) : (isset($row['l']) ? floatval($row['l']) : null);
                    $c = isset($row['close']) ? floatval($row['close']) : (isset($row['c']) ? floatval($row['c']) : null);
                    if ($x && $o!==null && $h!==null && $l!==null && $c!==null) {
                        $candles[] = [ 'x' => $x, 'o' => $o, 'h' => $h, 'l' => $l, 'c' => $c ];
                    }
                }
            }
            $cursor->modify('+7 days');
        }
    } catch (Throwable $e) {
        error_log("$logTag aggregatesRange error: " . $e->getMessage());
    }
    // 2) Si rango vacío, intentar intradía 1min
    if (empty($candles)) {
        try {
            $data = $finage->getIntradayData($symbol, '1min', $limit, 'forex');
            error_log("$logTag intraday 1min limit=$limit count=" . (is_array($data) ? count($data) : 0));
            if (is_array($data)) {
                foreach ($data as $row) {
                    $x = isset($row['time']) ? intval($row['time']) * 1000 : (isset($row['t']) ? intval($row['t']) : null);
                    $o = isset($row['open']) ? floatval($row['open']) : (isset($row['o']) ? floatval($row['o']) : null);
                    $h = isset($row['high']) ? floatval($row['high']) : (isset($row['h']) ? floatval($row['h']) : null);
                    $l = isset($row['low']) ? floatval($row['low']) : (isset($row['l']) ? floatval($row['l']) : null);
                    $c = isset($row['close']) ? floatval($row['close']) : (isset($row['c']) ? floatval($row['c']) : null);
                    if ($x && $o!==null && $h!==null && $l!==null && $c!==null) {
                        $candles[] = [ 'x' => $x, 'o' => $o, 'h' => $h, 'l' => $l, 'c' => $c ];
                    }
                }
            }
        } catch (Throwable $e) {
            error_log("$logTag finage intraday error: " . $e->getMessage());
        }
    }

    // 3) Fallback: si no hay datos, usar previous close para sembrar la última vela cerrada
    if (empty($candles)) {
        try {
            $apiKey = isset($GLOBALS['finage_api_key']) ? $GLOBALS['finage_api_key'] : (getenv('FINAGE_API_KEY') ?: '');
            $baseUrl = 'https://api.finage.co.uk';
            $url = $baseUrl . "/forex/prev-close/" . urlencode($symbol) . "?apikey=" . urlencode($apiKey);
            $ch = curl_init();
            curl_setopt_array($ch, [
                CURLOPT_URL => $url,
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_TIMEOUT => 10,
                CURLOPT_SSL_VERIFYPEER => false,
                CURLOPT_HTTPHEADER => ['Accept: application/json']
            ]);
            $response = curl_exec($ch);
            $code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
            curl_close($ch);
            if ($code === 200 && $response) {
                $json = json_decode($response, true);
                if (is_array($json) && isset($json['results']) && count($json['results']) > 0) {
                    $r = $json['results'][0];
                    $t = isset($r['t']) ? intval($r['t']) : null;
                    $o = isset($r['o']) ? floatval($r['o']) : null;
                    $h = isset($r['h']) ? floatval($r['h']) : null;
                    $l = isset($r['l']) ? floatval($r['l']) : null;
                    $c = isset($r['c']) ? floatval($r['c']) : null;
                    if ($t && $o!==null && $h!==null && $l!==null && $c!==null) {
                        $candles[] = [ 'x' => $t, 'o' => $o, 'h' => $h, 'l' => $l, 'c' => $c ];
                        error_log("$logTag prev-close used t=$t o=$o h=$h l=$l c=$c");
                    }
                }
            }
        } catch (Throwable $e) {
            error_log('candles.php prev-close error: ' . $e->getMessage());
        }
    }

    error_log("$logTag final candles count=" . count($candles) . " symbol=$symbol tf=1m limit=$limit");

    echo json_encode([ 'success' => true, 'data' => $candles, 'symbol' => $symbol, 'timeframe' => $timeframe ]);
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([ 'success' => false, 'error' => 'server_error', 'message' => $e->getMessage() ]);
}
