<?php
// Endpoint para logging de errores del frontend
session_start();
define('BASE_PATH', dirname(__DIR__, 2));
require_once BASE_PATH . '/database/config.php';
require_once BASE_PATH . '/database/connection.php';

// Configurar headers para CORS y JSON
header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, X-Requested-With');

// Manejar preflight OPTIONS request
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit();
}

// Solo permitir POST
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['error' => 'Método no permitido']);
    exit();
}

try {
    // Leer datos del request
    $input = file_get_contents('php://input');
    $data = json_decode($input, true);
    
    if (!$data) {
        throw new Exception('Datos inválidos');
    }
    
    // Validar campos requeridos
    $required_fields = ['type', 'message', 'timestamp', 'url'];
    foreach ($required_fields as $field) {
        if (!isset($data[$field])) {
            throw new Exception("Campo requerido faltante: $field");
        }
    }
    
    // Preparar datos para el log
    $log_entry = [
        'timestamp' => $data['timestamp'],
        'type' => $data['type'],
        'message' => $data['message'],
        'url' => $data['url'],
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
        'ip' => $_SERVER['REMOTE_ADDR'] ?? 'Unknown'
    ];
    
    // Agregar campos opcionales si existen
    if (isset($data['stack'])) {
        $log_entry['stack'] = $data['stack'];
    }
    if (isset($data['status'])) {
        $log_entry['status'] = $data['status'];
    }
    if (isset($data['statusText'])) {
        $log_entry['statusText'] = $data['statusText'];
    }
    
    // Determinar el archivo de log basado en el tipo
    $log_dir = BASE_PATH . '/logs';
    if (!is_dir($log_dir)) {
        mkdir($log_dir, 0755, true);
    }
    
    $log_file = $log_dir . '/frontend_errors.log';
    
    // Determinar si es crítico
    $critical_types = ['Error', 'Network Error', 'HTTP Error', 'Element Error', 'Data Error'];
    if (in_array($data['type'], $critical_types)) {
        $log_file = $log_dir . '/critical_errors.log';
    }
    
    // Formatear entrada del log
    $log_line = date('Y-m-d H:i:s') . ' [' . $data['type'] . '] ' . $data['message'];
    if (isset($data['url'])) {
        $log_line .= ' | URL: ' . $data['url'];
    }
    if (isset($data['stack'])) {
        $log_line .= ' | Stack: ' . substr($data['stack'], 0, 200) . '...';
    }
    $log_line .= ' | IP: ' . $log_entry['ip'] . PHP_EOL;
    
    // Escribir al archivo de log
    file_put_contents($log_file, $log_line, FILE_APPEND | LOCK_EX);
    
    // También escribir al log general
    $general_log = $log_dir . '/general.log';
    file_put_contents($general_log, $log_line, FILE_APPEND | LOCK_EX);
    
    // Respuesta exitosa
    echo json_encode([
        'success' => true,
        'message' => 'Error logged successfully',
        'log_file' => basename($log_file)
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'error' => 'Error al procesar el log: ' . $e->getMessage()
    ]);
    
    // Log del error del sistema de logging
    error_log('Error en log_error.php: ' . $e->getMessage());
}
?>