<?php
/**
 * Módulo Escritorios - Crear
 */

if (!isLoggedIn()) {
    header('Location: /index.php?module=auth&action=login');
    exit;
}

$db = getDB();
$message = '';
$error = '';

// Obtener usuarios para manager
$users = $db->select("SELECT id, username, first_name, last_name FROM users WHERE status = 'active' ORDER BY username");

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $name = trim($_POST['name'] ?? '');
    $description = trim($_POST['description'] ?? '');
    $manager_id = $_POST['manager_id'] ?? null;
    $status = $_POST['status'] ?? 'active';
    $working_hours_start = $_POST['working_hours_start'] ?? '09:00';
    $working_hours_end = $_POST['working_hours_end'] ?? '18:00';
    
    // Validaciones
    if (empty($name)) {
        $error = 'El nombre del escritorio es obligatorio.';
    } elseif (strlen($name) > 100) {
        $error = 'El nombre no puede exceder 100 caracteres.';
    } else {
        // Verificar si el nombre ya existe
        $existing = $db->selectOne(
            "SELECT id FROM desks WHERE name = ?",
            [$name]
        );
        
        if ($existing) {
            $error = 'Ya existe un escritorio con ese nombre.';
        } else {
            try {
                // Crear escritorio
                $desk_id = $db->execute(
                    "INSERT INTO desks (name, description, manager_id, status, working_hours_start, working_hours_end, created_at) 
                     VALUES (?, ?, ?, ?, ?, ?, NOW())",
                    [$name, $description, $manager_id ?: null, $status, $working_hours_start, $working_hours_end]
                );
                
                // Registrar actividad
                $db->execute(
                    "INSERT INTO lead_activities (type, subject, description, created_by, created_at) 
                     VALUES ('system', 'Escritorio Creado', ?, ?, NOW())",
                    ["Escritorio creado: $name", $_SESSION['user_id']]
                );
                
                $message = 'Escritorio creado exitosamente.';
                
                // Limpiar formulario
                $name = $description = '';
                $manager_id = null;
                
            } catch (Exception $e) {
                $error = 'Error al crear el escritorio: ' . $e->getMessage();
            }
        }
    }
}
?>

<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Crear Escritorio - <?php echo APP_NAME; ?></title>
    <link rel="stylesheet" href="/test/simple_crm/public/style.css">
</head>
<body>
    <?php include BASE_PATH . '/modules/shared/header.php'; ?>
    
    <div class="container">
        <div class="card">
            <div class="card-header">
                <h1 class="card-title">Crear Nuevo Escritorio</h1>
                <a href="?module=desks" class="btn btn-secondary">Volver al Listado</a>
            </div>
            
            <?php if ($error): ?>
                <div class="alert alert-error"><?php echo htmlspecialchars($error); ?></div>
            <?php endif; ?>
            
            <?php if ($message): ?>
                <div class="alert alert-success"><?php echo htmlspecialchars($message); ?></div>
            <?php endif; ?>
            
            <form method="POST" class="form">
                <div class="form-row">
                    <div class="form-group">
                        <label for="name">Nombre del Escritorio *</label>
                        <input type="text" id="name" name="name" 
                               value="<?php echo htmlspecialchars($name ?? ''); ?>" 
                               required maxlength="100"
                               placeholder="Ej: Ventas Madrid, Soporte Técnico, etc.">
                    </div>
                    
                    <div class="form-group">
                        <label for="manager_id">Manager</label>
                        <select id="manager_id" name="manager_id">
                            <option value="">Seleccionar manager...</option>
                            <?php foreach ($users as $user): ?>
                                <option value="<?php echo $user['id']; ?>"
                                        <?php echo ($manager_id ?? '') == $user['id'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($user['username'] . ' - ' . $user['first_name'] . ' ' . $user['last_name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                </div>
                
                <div class="form-group">
                    <label for="description">Descripción</label>
                    <textarea id="description" name="description" rows="3" 
                              placeholder="Descripción del escritorio, objetivos, etc."><?php echo htmlspecialchars($description ?? ''); ?></textarea>
                </div>
                
                <div class="form-row">
                    <div class="form-group">
                        <label for="working_hours_start">Hora de Inicio</label>
                        <input type="time" id="working_hours_start" name="working_hours_start" 
                               value="<?php echo htmlspecialchars($working_hours_start ?? '09:00'); ?>">
                    </div>
                    
                    <div class="form-group">
                        <label for="working_hours_end">Hora de Fin</label>
                        <input type="time" id="working_hours_end" name="working_hours_end" 
                               value="<?php echo htmlspecialchars($working_hours_end ?? '18:00'); ?>">
                    </div>
                    
                    <div class="form-group">
                        <label for="status">Estado</label>
                        <select id="status" name="status">
                            <option value="active" <?php echo ($status ?? 'active') === 'active' ? 'selected' : ''; ?>>Activo</option>
                            <option value="inactive" <?php echo ($status ?? '') === 'inactive' ? 'selected' : ''; ?>>Inactivo</option>
                        </select>
                    </div>
                </div>
                
                <div class="form-actions">
                    <button type="submit" class="btn btn-success">Crear Escritorio</button>
                    <a href="?module=desks" class="btn btn-secondary">Cancelar</a>
                </div>
            </form>
        </div>
        
        <!-- Información adicional -->
        <div class="card">
            <div class="card-header">
                <h3>Información sobre Escritorios</h3>
            </div>
            <div class="info-grid">
                <div class="info-item">
                    <h4>¿Qué es un Escritorio?</h4>
                    <p>Un escritorio es una unidad organizacional que agrupa usuarios y leads por área, región o especialidad.</p>
                </div>
                <div class="info-item">
                    <h4>Manager</h4>
                    <p>El manager es responsable de supervisar el escritorio y sus usuarios asignados.</p>
                </div>
                <div class="info-item">
                    <h4>Horarios de Trabajo</h4>
                    <p>Define las horas operativas del escritorio para la asignación automática de leads.</p>
                </div>
            </div>
        </div>
    </div>
    
    <script src="/public/script.js"></script>
    <style>
        .info-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 1rem;
            padding: 1rem;
        }
        
        .info-item h4 {
            color: #007bff;
            margin-bottom: 0.5rem;
            font-size: 1rem;
        }
        
        .info-item p {
            color: #6c757d;
            font-size: 0.9rem;
            line-height: 1.4;
            margin: 0;
        }
    </style>
</body>
</html>