<?php
/**
 * Script para inicializar la base de datos con esquema y datos de ejemplo
 */

require_once 'database/connection.php';

try {
    $db = getDB();
    $pdo = $db->getConnection();
    
    echo "<h2>Inicializando Base de Datos...</h2>";
    
    // Leer y ejecutar schema.sql
    echo "<h3>1. Ejecutando esquema de base de datos...</h3>";
    $schema_sql = file_get_contents('database/schema.sql');
    
    if ($schema_sql === false) {
        throw new Exception("No se pudo leer el archivo schema.sql");
    }
    
    // Dividir en statements individuales
    $statements = explode(';', $schema_sql);
    $executed = 0;
    
    foreach ($statements as $statement) {
        $statement = trim($statement);
        if (!empty($statement)) {
            try {
                $pdo->exec($statement);
                $executed++;
            } catch (PDOException $e) {
                // Ignorar errores de tablas que ya existen
                if (strpos($e->getMessage(), 'already exists') === false) {
                    echo "<p style='color: orange;'>Warning: " . $e->getMessage() . "</p>";
                }
            }
        }
    }
    
    echo "<p style='color: green;'>✓ Esquema ejecutado: $executed statements</p>";
    
    // Leer y ejecutar seed_sample_data.sql
    echo "<h3>2. Insertando datos de ejemplo...</h3>";
    $seed_sql = file_get_contents('database/seed_sample_data.sql');
    
    if ($seed_sql === false) {
        throw new Exception("No se pudo leer el archivo seed_sample_data.sql");
    }
    
    // Dividir en statements individuales
    $statements = explode(';', $seed_sql);
    $executed = 0;
    
    foreach ($statements as $statement) {
        $statement = trim($statement);
        if (!empty($statement) && !str_starts_with($statement, '--')) {
            try {
                $pdo->exec($statement);
                $executed++;
            } catch (PDOException $e) {
                // Ignorar errores de duplicados
                if (strpos($e->getMessage(), 'Duplicate entry') === false) {
                    echo "<p style='color: orange;'>Warning: " . $e->getMessage() . "</p>";
                }
            }
        }
    }
    
    echo "<p style='color: green;'>✓ Datos de ejemplo insertados: $executed statements</p>";
    
    // Verificar usuarios creados
    echo "<h3>3. Verificando usuarios creados...</h3>";
    $users = $db->select('SELECT id, username, email, first_name, last_name, status FROM users');
    
    if (!empty($users)) {
        echo "<p style='color: green;'>✓ Usuarios creados exitosamente:</p>";
        echo "<ul>";
        foreach ($users as $user) {
            echo "<li><strong>" . htmlspecialchars($user['username']) . "</strong> - " . 
                 htmlspecialchars($user['first_name'] . ' ' . $user['last_name']) . 
                 " (" . htmlspecialchars($user['email']) . ")</li>";
        }
        echo "</ul>";
        
        echo "<h3>4. Credenciales de acceso:</h3>";
        echo "<div style='background: #f0f0f0; padding: 15px; border-radius: 5px;'>";
        echo "<p><strong>Todas las contraseñas son:</strong> <code>password</code></p>";
        echo "<p><strong>Usuarios disponibles:</strong></p>";
        echo "<ul>";
        foreach ($users as $user) {
            echo "<li>Usuario: <code>" . htmlspecialchars($user['username']) . "</code> o Email: <code>" . htmlspecialchars($user['email']) . "</code></li>";
        }
        echo "</ul>";
        echo "</div>";
        
    } else {
        echo "<p style='color: red;'>✗ No se crearon usuarios</p>";
    }
    
    echo "<h2 style='color: green;'>✅ Inicialización completada</h2>";
    echo "<p><a href='index.php'>Ir al sistema</a></p>";
    
} catch (Exception $e) {
    echo "<h2 style='color: red;'>❌ Error durante la inicialización</h2>";
    echo "<p style='color: red;'>Error: " . $e->getMessage() . "</p>";
    echo "<p>Verifica que:</p>";
    echo "<ul>";
    echo "<li>MySQL esté ejecutándose</li>";
    echo "<li>La base de datos 'simple_crm' exista</li>";
    echo "<li>Las credenciales de conexión sean correctas</li>";
    echo "</ul>";
}
?>