<?php
/**
 * Importador de Base de Datos a Producción
 * - Crea la BD `spin2pay_profixcrm`
 * - Aplica el esquema desde schema.sql (ajustando el nombre de BD)
 * - Asegura rol Admin con todos los permisos
 * - Crea/actualiza usuario admin con clave `admin123` y asigna rol Admin
 *
 * Uso (desde CLI):
 *   php simple_crm/database/import_production.php
 */

// Preferir configuración local si existe; en su defecto, producción
if (file_exists(__DIR__ . '/config.local.php')) {
    require_once __DIR__ . '/config.local.php';
} elseif (file_exists(__DIR__ . '/config.prod.php')) {
    require_once __DIR__ . '/config.prod.php';
}
require_once __DIR__ . '/config.php';

// Conectar sin seleccionar base específica para poder crearla
$dsn = "mysql:host=" . DB_HOST . ";port=" . DB_PORT . ";charset=" . DB_CHARSET;
try {
    $pdo = new PDO($dsn, DB_USER, DB_PASS, [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false
    ]);
} catch (PDOException $e) {
    fwrite(STDERR, "Error de conexión: " . $e->getMessage() . "\n");
    exit(1);
}

$targetDb = 'spin2pay_profixcrm';

function runSqlBatch(PDO $pdo, string $sql, string $ctx = 'batch') {
    // Eliminar comentarios y dividir por ';' simples (no hay procedimientos)
    $clean = preg_replace('/^\s*--.*$/m', '', $sql);
    $stmts = array_filter(array_map('trim', explode(';', $clean)), function($s){ return $s !== ''; });
    foreach ($stmts as $stmt) {
        try {
            $pdo->exec($stmt);
        } catch (PDOException $e) {
            throw new RuntimeException("Fallo en $ctx: " . $e->getMessage() . "\nSQL: " . $stmt);
        }
    }
}

// 1) Reiniciar BD y usarla (drop + create para entorno limpio)
echo "[1/4] Reiniciando base de datos `$targetDb`...\n";
$pdo->exec("DROP DATABASE IF EXISTS `$targetDb`");
$pdo->exec("CREATE DATABASE `$targetDb` CHARACTER SET utf8mb4 COLLATE utf8mb4_unicode_ci");
$pdo->exec("USE `$targetDb`");

// 2) Cargar schema.sql ajustando nombre de BD
echo "[2/4] Aplicando esquema desde schema.sql...\n";
$schemaPath = __DIR__ . '/schema.sql';
if (!file_exists($schemaPath)) {
    fwrite(STDERR, "No se encuentra schema.sql en $schemaPath\n");
    exit(1);
}
$schema = file_get_contents($schemaPath);
// Remover/neutralizar creación y USE de 'simple_crm'
$schema = preg_replace('/CREATE DATABASE IF NOT EXISTS\s+simple_crm.*?;\s*/is', '', $schema);
$schema = preg_replace('/USE\s+simple_crm\s*;\s*/is', '', $schema);
// Ejecutar el resto del esquema en la BD activa
runSqlBatch($pdo, $schema, 'schema.sql');

// 3) Asegurar rol admin y permisos completos
echo "[3/4] Asegurando rol Admin y permisos...\n";
// Crear rol admin si no existe
$pdo->exec("INSERT IGNORE INTO roles (name, display_name, description, is_system, is_active) VALUES ('admin','Administrador','Acceso completo al sistema', TRUE, TRUE)");
// Insertar permisos base si el archivo de seed existe
$seedPermsPath = __DIR__ . '/seed_permissions.sql';
if (file_exists($seedPermsPath)) {
    $seed = file_get_contents($seedPermsPath);
    runSqlBatch($pdo, $seed, 'seed_permissions.sql');
}
// Asegurar que el rol admin tenga todos los permisos
$pdo->exec("INSERT IGNORE INTO role_permissions (role_id, permission_id) SELECT r.id, p.id FROM roles r, permissions p WHERE r.name = 'admin'");

// 4) Crear/actualizar usuario admin con clave admin123 y asignar rol Admin
echo "[4/4] Creando/actualizando usuario admin y asignando rol...\n";
$hash = password_hash('admin123', PASSWORD_DEFAULT);
// Crear usuario si no existe
$pdo->exec("INSERT IGNORE INTO users (username, email, first_name, last_name, password_hash, status, created_at)
VALUES ('admin', 'admin@profixcrm.local', 'Admin', 'Sistema', '$hash', 'active', NOW())");
// Si existe, actualizar su password al hash solicitado
$stmt = $pdo->prepare("UPDATE users SET password_hash = :h, status = 'active' WHERE username = 'admin'");
$stmt->execute([':h' => $hash]);
// Asignar rol admin al usuario admin
$pdo->exec("INSERT IGNORE INTO user_roles (user_id, role_id)
SELECT u.id, r.id FROM users u, roles r WHERE u.username = 'admin' AND r.name = 'admin'");

echo "Importación completada correctamente en la base `$targetDb`.\n";
echo "Nota: Cambia la contraseña del usuario 'admin' tras el primer acceso.\n";

?>